Imports EJRandomOrgConsumer.StructuresAndEnums
Imports System.ComponentModel

''' <summary>
''' Provides methods and properties for consuming the HTTP service at random.org that generates random sequences integers within a range.
''' </summary>
''' <remarks></remarks>
Public Class SequenceGenerator
    Inherits BaseGenerator


    Private Const MaxQuantity As Short = 10000
    Private Const MinValue As Integer = -1000000000
    Private Const MaxValue As Integer = 1000000000

    Private _URL As String = "http://www.random.org/sequences/"
    Private _Range As New ValueRange(MinValue, MaxValue)


#Region " Properties "


    ''' <summary>
    ''' Gets or sets a value specifying the URL providing the service.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    <Browsable(True), DefaultValue("http://www.random.org/sequences/"), Description("Gets or sets a value specifying the URL providing the service.")> _
     Public Property URL() As String
        Get
            Return _URL
        End Get
        Set(ByVal value As String)
            _URL = value
        End Set
    End Property


    ''' <summary>
    ''' Gets or sets the minimum and maximum value of each randomly-generated integer.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    <Browsable(True), Description("Gets or sets the minimum and maximum values (inclusive) comprising the sequence of integers.")> _
    Public Property Range() As ValueRange
        Get
            Return _Range
        End Get
        Set(ByVal value As ValueRange)
            If value.MinValue < MinValue Then
                Throw New Exception("The minimum allowed value is " & MinValue.ToString("#,###") & ".")
            ElseIf value.MaxValue > MaxValue Then
                Throw New Exception("The minimum allowed value is " & MaxValue.ToString("#,###") & ".")
            ElseIf value.MinValue >= value.MaxValue Then
                Throw New Exception("The minimum value must be less than the maximum value.")
            ElseIf (value.MaxValue - value.MinValue) + 1 > MaxQuantity Then
                Throw New Exception("The quantity of integers comprising the sequence cannot exceed " & MaxValue.ToString("#,###") & ".")
            Else
                _Range = value
            End If
        End Set
    End Property


#End Region


    Public Sub New()

    End Sub


    Public Overrides Sub Dispose()
        MyBase.Dispose()


        _Range = Nothing


    End Sub


    ''' <summary>
    ''' Connects to the remote service, requests a sequence of randomly-generated integers that meet the critera specified by this instance's property values, and returns the response as an array of integers.
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Function GetSequence() As Integer()


        Dim NewIntegers() As Integer = Nothing

        Dim FullURL As String = ""
        Dim Response As String = ""
        Dim ResponseElements() As String = Nothing
        Dim TempResponseElement As String


        Try

            'Initialize the return array.
            ReDim NewIntegers(-1)

            'Build the full URL path, including all of the variable name/value pairs.
            FullURL = GetFullURL()

            'Request the data from the server.
            Response = RemoteData.GetPageSimple(FullURL, ConnectionTimeout)

            'Does the response contain an error (if so, it will contain a colon, as per the specification at http://random.org/clients/http/ )
            If Response.Contains(":") Then
                Throw New Exception(Response)
            Else

                'Normalize linefeeds/carriage returns.
                Response = Response.Replace(vbCrLf, vbLf).Replace(vbCr, vbLf)

                'Remove trailing linefeeds.
                While Response.EndsWith(vbLf)
                    Response = Response.Substring(0, Response.Length - 1)
                End While

                'Split the response into an array of numeric strings.
                ResponseElements = Split(Response, vbLf)

                'Add each response element to our final array of integers.
                For Each TempResponseElement In ResponseElements
                    If IsNumeric(TempResponseElement) Then
                        ReDim Preserve NewIntegers(NewIntegers.Length)
                        NewIntegers(NewIntegers.Length - 1) = CType(TempResponseElement, Integer)
                    End If
                Next

            End If

        Catch ex As Exception

            Throw

        Finally

            ResponseElements = Nothing

        End Try


        Return NewIntegers


    End Function


    Private Function GetFullURL() As String


        Dim sbFullURL As New System.Text.StringBuilder()
        Dim FullURL As String = ""


        Try

            sbFullURL.Append(URL)

            sbFullURL.Append("?min=")
            sbFullURL.Append(Range.MinValue.ToString())

            sbFullURL.Append("&max=")
            sbFullURL.Append(Range.MaxValue.ToString())

            sbFullURL.Append("&format=")
            sbFullURL.Append("plain")

            sbFullURL.Append("&rnd=")
            Select Case RandomizationMethod
                Case RandomizationMethods.IdentifierBased
                    sbFullURL.Append("id.")
                    sbFullURL.Append(RandomizationIdentifier)
                Case RandomizationMethods.DateBased
                    sbFullURL.Append("date.")
                    sbFullURL.Append(RandomizationDate.ToString("yyyy-MM-dd"))
                Case Else
                    sbFullURL.Append("new")
            End Select

            FullURL = sbFullURL.ToString()

        Catch ex As Exception

            Throw

        Finally

            sbFullURL = Nothing

        End Try


        Return FullURL


    End Function


End Class
